/*
 * Copyright (C) Jan 2006 Mellanox Technologies Ltd. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 *  MultiFile.h - Multifile (read-only file with subsections) class
 *
 *  Version: $Id: MultiFile.h 2752 2006-01-19 14:40:17Z mst $
 *
 */

#ifndef MULTIFILE_H
#define MULTIFILE_H

#include <string>
#include <map>
#include <list>
#include <vector>
#include "compatibility.h"
#include "utils.h"

// Image file stamps:
#define MAGIC_STRING_FORMAT "<!-- MT%d Firmware image"
#define MAGIC_STRING_PREFIX "<!-- MT"

enum DeviceType {
    UNKNOWN = 0,
    MT47396 = 47396, // ANAFA2
    MT23108 = 23108, // TAVOR
    MT25208 = 25208, // ARBEL 
    MT25218 = 25218, // ARBEL MEMFREE
    MT24204 = 24204, // SINAI 4x
    MT25204 = 25204  // SINAI 8x
};

class MSection
{
public:
    MSection(const char *n) : name(n) {}

    // crstart values boot sections - these are not real cr-addresses
    enum CrStartBoot {BOOT0 = 0, BOOT2 = 2, BOOT3 = 3};

    std::string              name;        // Section name
    bool                     bin;         // Is section binary
    u_int32_t                start_line;  // First line number
    u_int32_t                addr;        // For binary sections - start address
    u_int32_t                start;       // For binary sections - start run
    u_int32_t                end;         // For binary sections
    u_int32_t                crstart;     // For binary sections
    std::vector<u_int8_t>    data;        // For binary sections - data
    std::vector<std::string> lines;       // For ASCII sections - input lines
};

class MultiFile : public ErrMsg
{
public:
    MultiFile(DeviceType dev_type);
    ~MultiFile();

    const char             *name() const      { return _fname;      };
    void                   clear();
    bool                   read(const char *fname);
    MSection               *operator[](const char *name) const;
    std::list<std::string> sections() const;
    std::list<std::string> non_boot_sections() const;
    DeviceType             DevType() const {return _dev_type;}

private:
    std::map<std::string, MSection*> _sects;
    char                             *_fname;
    char                              _magic[64];
    DeviceType                       _dev_type;

    bool get_section_header   (int line, const std::string& header, const char* attr, u_int32_t& val);
    bool get_section_header_cr(int line, const std::string& header, const char* attr, u_int32_t& val);
    bool get_section_header   (int line, const std::string& header, const char* attr, std::string&    val);

};

#endif
